local e = defines.events

local function setup_storage()
    storage.entity_events = storage.entity_events or {}
    storage.stack_events = storage.stack_events or {}
    storage.attempted_tick = storage.attempted_tick or {}
end

script.on_init(setup_storage)
script.on_configuration_changed(setup_storage)

---@param inventory LuaInventory?
local function get_validated_bar(inventory)
    if not inventory then return end
    if not inventory.supports_bar() then return end
    local bar = inventory.get_bar()
    if bar > #inventory then return end
    return bar
end

---@param entity LuaEntity
---@return table<uint, LuaEntity>
local function disable_target_entities(entity)
    local box = entity.bounding_box
    local lt, rb = box.left_top, box.right_bottom
    lt = {x = math.floor(lt.x) - 3, y = math.floor(lt.y) - 3}
    rb = {x = math.ceil(rb.x) + 3, y = math.ceil(rb.y) + 3}
    local area = {left_top = lt, right_bottom = rb}

    local entities = entity.surface.find_entities_filtered{area = area, type = {"inserter", "loader", "loader-1x1"}}
    for i, target_entity in pairs(entities) do
        if target_entity.active then
            target_entity.active = false
        else
            entities[i] = nil
        end
    end

    return entities
end

local function restore_state(data)
    if not data then return end

    if data.inventory.valid then
        data.inventory.set_bar(data.bar)
    end

    for _, entity in pairs(data.entities) do
        if entity.valid then
            entity.active = true
        end
    end

    return true
end

---@param event EventData.CustomInputEvent
script.on_event({"fii-fast-entity-split", "fii-fast-entity-transfer"}, function(event)
    local player = game.get_player(event.player_index) --[[@as LuaPlayer]]
    local entity = player.selected
    if not entity then return end

    local cursor = player.cursor_stack
    if not (cursor and cursor.valid_for_read) then return end

    local inventory = entity.get_inventory(defines.inventory.chest)

    local bar = get_validated_bar(inventory)
    if not bar then return end ---@cast inventory -?

    if not player.mod_settings["fii-always-force"].value then
        if inventory.get_insertable_count(cursor.name) > 0 then return end
    end

    inventory.set_bar()
    if inventory.get_insertable_count(cursor.name) == 0 then
        inventory.set_bar(bar)
        return
    end

    local entities = disable_target_entities(entity)
    storage.entity_events[event.player_index] = {
        inventory = inventory,
        bar = bar,
        entities = entities,
        tick = event.tick,
    }
end)

script.on_event(e.on_player_cursor_stack_changed, function(event)
    if not restore_state(storage.entity_events[event.player_index]) then return end
    storage.entity_events[event.player_index] = nil
end)

---@param event EventData.CustomInputEvent
script.on_event({"fii-stack-split", "fii-stack-transfer", "fii-inventory-split", "fii-inventory-transfer"}, function(event)
    local player = game.get_player(event.player_index) --[[@as LuaPlayer]]
    local entity = player.opened
    if not entity then return end
    if player.opened_gui_type ~= defines.gui_type.entity then return end ---@cast entity LuaEntity

    local selected_prototype = event.selected_prototype
    if not selected_prototype then return end
    if selected_prototype.base_type ~= "item" then return end

    local inventory = entity.get_inventory(defines.inventory.chest)
    local bar = get_validated_bar(inventory)
    if not bar then return end ---@cast inventory -?

    if not player.mod_settings["fii-always-force"].value then
        if not storage.attempted_tick[event.player_index] then
            storage.attempted_tick[event.player_index] = event.tick
            if inventory.get_insertable_count(selected_prototype.name) > 0 then return end
        end
    end

    inventory.set_bar()
    if inventory.get_insertable_count(selected_prototype.name) == 0 then
        inventory.set_bar(bar)
        return
    end

    local entities = disable_target_entities(entity)
    storage.stack_events[event.player_index] = {
        inventory = inventory,
        bar = bar,
        entities = entities,
        tick = event.tick,
    }
end)

script.on_event(e.on_player_main_inventory_changed, function(event)
    if not restore_state(storage.stack_events[event.player_index]) then return end
    storage.stack_events[event.player_index] = nil
end)

script.on_event(e.on_tick, function(event)
    for index, data in pairs(storage.stack_events) do
        if data.tick == event.tick and restore_state(data) then
            storage.stack_events[index] = nil
        end
    end

    for index, tick in pairs(storage.attempted_tick) do
        if event.tick - tick >= 30 then
            storage.attempted_tick[index] = nil
        end
    end
end)