--TODO: Autosort containers while open?
--TODO: Notify the player that the game's autosort has to be disabled either on init or on first autosort toggle.
--TODO: Reimplement partial inventory sorting using a dummy chest probably


------- Init options caching -------

local options_cache

local function load_options_cache(player_index)
	local options = settings.get_player_settings(player_index)
	return {
		sort_buttons = options['manual-inventory-sort-buttons'].value,
		sort_on_open = options['manual-inventory-sort-on-open'].value,
		sort_self_on_open = options['manual-inventory-sort-self-on-open'].value,
		auto_sort = options['manual-inventory-auto-sort'].value,
		allow_sorting_remotely = options['manual-inventory-allow-sorting-remotely'].value,
	}
end

local function init_options_cache(clear)
	if clear then
		storage.options_cache = nil
		options_cache = nil
	end

	storage.options_cache = storage.options_cache or {}
	options_cache = options_cache or setmetatable(storage.options_cache, {
		__index = function(self, player_index)
			local cache = load_options_cache(player_index)
			self[player_index] = cache
			return cache
		end,
	})
end

local function update_options_cache(player_index)
	if player_index then options_cache[player_index] = nil
	else options_cache = {}; end
end


script.on_init(init_options_cache)
script.on_load(init_options_cache)
script.on_configuration_changed(init_options_cache)

script.on_event(defines.events.on_runtime_mod_setting_changed, function(event)
	if event.player_index then update_options_cache(event.player_index)
	else init_options_cache(true); end
end)


------- Constants -------

local CONTROLLERS = defines.controllers

local function find_main_sortable_inventory(player)
	local character = player.character
	if character ~= nil then return character.get_inventory(defines.inventory.character_main); end
	return player.get_main_inventory()
end

local function has_sortable_trash(player)
	return player.controller_type == CONTROLLERS.character
end

local function can_sort_entities(player)
	local controller = player.controller_type
	return (
		controller == CONTROLLERS.character
		or controller == CONTROLLERS.god
		or controller == CONTROLLERS.remote and options_cache[player.index].allow_sorting_remotely
	) and player.opened_gui_type == defines.gui_type.entity
end

local SORTABLE_ENTITY_INVENTORIES = {
	['container'] = {defines.inventory.chest},
	['logistic-container'] = {defines.inventory.chest, defines.inventory.logistic_container_trash},
	['linked-container'] = {defines.inventory.chest},
	['car'] = {defines.inventory.car_trunk},
	['cargo-wagon'] = {defines.inventory.cargo_wagon},
	['spider-vehicle'] = {defines.inventory.spider_trunk, defines.inventory.spider_trash},
}


------- Some helper functions -------

local function sort_player(index)
	local inventory = find_main_sortable_inventory(game.get_player(index))
	if not inventory then return; end
	inventory.sort_and_merge()
end

local function sort_player_trash(index)
	local player = game.get_player(index)
	-- God controller doesn't have trash slots, so this is just hardcoded here for now.
	if not has_sortable_trash(player) then return; end
	player.get_inventory(defines.inventory.character_trash).sort_and_merge()
end

local function sort_opened(index)
	local player = game.get_player(index)
	if not can_sort_entities(player) then return; end
	local entity = player.opened
	local inventories = SORTABLE_ENTITY_INVENTORIES[entity.type]
	if not inventories then return; end
	for _, inventory_id in pairs(inventories) do
		entity.get_inventory(inventory_id).sort_and_merge()
	end
end


------- Sorting event handling -------

script.on_event('manual-inventory-sort', function(event) sort_player(event.player_index); end)
script.on_event('manual-inventory-sort-opened', function(event) sort_opened(event.player_index); end)

script.on_event('manual-inventory-auto-sort-toggle', function(event)
	local player = game.get_player(event.player_index)
	local options = settings.get_player_settings(player)

	local value = not options['manual-inventory-auto-sort'].value
	options['manual-inventory-auto-sort'] = {value = value}
	update_options_cache(player.index) -- This can be removed if the above line ever starts raising an event properly

	player.print{value and 'manual-inventory-auto-sort-on' or 'manual-inventory-auto-sort-off'}
	if value then sort_player(player.index); end
end)

script.on_event(defines.events.on_player_main_inventory_changed, function(event)
	if options_cache[event.player_index].auto_sort then sort_player(event.player_index); end
end)


------- Gui stuff -------

local function sort_buttons_gui(player_index, closing)
	local player = game.get_player(player_index)
	local frame = player.gui.left['manual-inventory-sort-buttons']
	if not closing and (player.opened or player.opened_self) then
		if not frame then
			local has_sortable_inventory = find_main_sortable_inventory(player) ~= nil
			local has_sortable_trash = has_sortable_trash(player)
			local can_sort_opened = can_sort_entities(player) and SORTABLE_ENTITY_INVENTORIES[player.opened.type]
			if not has_sortable_inventory and not has_sortable_trash and not can_sort_opened then return; end

			frame = player.gui.left.add{
				type = 'frame',
				name = 'manual-inventory-sort-buttons',
				direction = 'vertical',
				caption = {'manual-inventory-gui-sort-title'},
			}
			if has_sortable_inventory then
				frame.add{type='button', name='manual-inventory-sort-player', caption={'manual-inventory-gui-sort_player'}}
			end
			if has_sortable_trash then
				frame.add{type='button', name='manual-inventory-sort-player-trash', caption={'manual-inventory-gui-sort_player_trash'}}
			end
			if can_sort_opened then
				frame.add{type='button', name='manual-inventory-sort-opened', caption=player.opened.localised_name}
			end
		end
	elseif frame then frame.destroy(); end
end


------- Gui events -------

script.on_event(defines.events.on_gui_opened, function(event)
	local options = options_cache[event.player_index]
	if options.sort_buttons then sort_buttons_gui(event.player_index); end
	if options.sort_on_open then sort_opened(event.player_index); end
	if options.sort_self_on_open then sort_player(event.player_index); end
end)
script.on_event(defines.events.on_gui_closed, function(event) sort_buttons_gui(event.player_index, true); end)

script.on_event(defines.events.on_gui_click, function(event)
	if event.element.name == 'manual-inventory-sort-player' then sort_player(event.player_index)
	elseif event.element.name == 'manual-inventory-sort-player-trash' then sort_player_trash(event.player_index)
	elseif event.element.name == 'manual-inventory-sort-opened' then sort_opened(event.player_index); end
end)
